<?php
/**
 * Global Color Migration
 *
 * Handles the migration of global colors from CSS variable format to $variable() syntax.
 *
 * @since ??
 *
 * @package Divi
 */

namespace ET\Builder\Migration;

use ET\Builder\Packages\Conversion\Utils\ConversionUtils;
use ET\Builder\FrontEnd\BlockParser\BlockParserStore;
use ET\Builder\FrontEnd\BlockParser\BlockParserBlock;
use ET\Builder\VisualBuilder\Saving\SavingUtility;
use ET\Builder\Migration\MigrationContext;

/**
 * Global Color Migration Class.
 *
 * @since ??
 */
class GlobalColorMigration implements MigrationInterface {

	/**
	 * The migration name.
	 *
	 * @since ??
	 *
	 * @var string
	 */
	private static $_name = 'globalColor.v1';

	/**
	 * The Global Color migration release version string.
	 *
	 * @since ??
	 *
	 * @var string
	 */
	private static $_release_version = '5.0.0-public-alpha.17.1';

	/**
	 * CSS Variable pattern for global colors.
	 *
	 * @since ??
	 *
	 * @var string
	 */
	private static $_css_variable_pattern = '/var\(--(gcid-[0-9a-z-]+)\)/';

	/**
	 * Run the Global Color migration.
	 *
	 * For the time being, this migration only run when the content is loaded in Visual Builder.
	 * No need for migrating render on Frontend because existing Global Color mechanism will handle it.
	 * Migration on content import will be added once https://github.com/elegantthemes/Divi/issues/43481 is addressed.
	 *
	 * @since ??
	 */
	public static function load(): void {
		add_action( 'et_fb_load_raw_post_content', [ __CLASS__, 'migrate_vb_content' ], 10, 2 );
		add_filter( 'divi_visual_builder_rest_divi_library_load', [ __CLASS__, 'migrate_rest_divi_library_load' ] );
	}

	/**
	 * Get the migration name.
	 *
	 * @since ??
	 *
	 * @return string The migration name.
	 */
	public static function get_name() {
		return self::$_name;
	}

	/**
	 * Migrate the Visual Builder content.
	 *
	 * @since ??
	 *
	 * @param string $content The content to migrate.
	 * @return string The migrated content.
	 */
	public static function migrate_vb_content( $content ) {
		return self::_migrate_the_content( $content );
	}

	/**
	 * Migrate the REST Divi Library load response.
	 *
	 * @since ??
	 *
	 * @param WP_Post $post The post object.
	 *
	 * @return WP_Post The migrated post object.
	 */
	public static function migrate_rest_divi_library_load( $post ) {

		// If post content exist, run migration on loaded Divi Library's content.
		if ( isset( $post->post_content ) ) {
			$post->post_content = self::_migrate_the_content( $post->post_content );
		}

		return $post;
	}

	/**
	 * Migrate the content.
	 *
	 * @since ??
	 *
	 * @param string $content The content to migrate.
	 *
	 * @return string The migrated content.
	 */
	private static function _migrate_the_content( $content ) {
		// Ensure the content is wrapped by wp:divi/placeholder if not empty.
		// This is needed because later `SavingUtility::serialize_sanitize_blocks` is used and it uses
		// `get_comment_delimited_block_content()` that will skip the direct child of `divi/root`, causing $content
		// without placeholder block to be broken: only the first row gets rendered, the rest is gone.
		// See: https://elegantthemes.slack.com/archives/C01CW343ZJ9/p1750440835358589?thread_ts=1750394948.448129&cid=C01CW343ZJ9.
		$is_wrapped = '' !== $content &&
			strpos( $content, '<!-- wp:divi/placeholder -->' ) === 0 &&
			strpos( $content, '<!-- /wp:divi/placeholder -->' ) !== false;

		if ( ! $is_wrapped ) {
			$content = "<!-- wp:divi/placeholder -->\n" . $content . "\n<!-- /wp:divi/placeholder -->";
		}

		// Start migration context to prevent global layout expansion during migration.
		MigrationContext::start();

		try {
			$flat_objects = ConversionUtils::parseSerializedPostIntoFlatModuleObject( $content );

			foreach ( $flat_objects as $module_id => $module_data ) {
				// Skip Global Color migration on shortcode module because there's no D5 Global Color in shortcode module.
				$is_shortcode_module = 'divi/shortcode-module' === $module_data['name'];

				// Check if module needs migration based on version comparison.
				$builder_version = $module_data['props']['attrs']['builderVersion'] ?? '0.0.0';

				if ( ! $is_shortcode_module && version_compare( $builder_version, self::$_release_version, '<' ) ) {
					$migrated_attrs = self::_migrate_module_attributes( $module_data['props']['attrs'] );

					if ( $migrated_attrs !== $module_data['props']['attrs'] ) {
						// Update builder version and apply migrated attributes.
						$flat_objects[ $module_id ]['props']['attrs'] = array_merge(
							$migrated_attrs,
							[ 'builderVersion' => self::$_release_version ]
						);
					}
				}
			}

			// Serialize the flat objects back into the content.
			$blocks      = self::_flat_objects_to_blocks( $flat_objects );
			$new_content = SavingUtility::serialize_sanitize_blocks( $blocks );

			// Reset the block parser store and order index to avoid conflicts with rendering.
			BlockParserStore::reset();
			BlockParserBlock::reset_order_index();

			return $new_content;
		} finally {
			// Always end migration context, even if an exception occurs.
			MigrationContext::end();
		}
	}

	/**
	 * Migrate module attributes by converting CSS variables to $variable() syntax.
	 *
	 * @since ??
	 *
	 * @param array $attrs Module attributes.
	 *
	 * @return array Migrated attributes.
	 */
	private static function _migrate_module_attributes( array $attrs ): array {
		return self::_migrate_attributes_recursive( $attrs );
	}

	/**
	 * Recursively migrate attributes to convert global color CSS variables.
	 *
	 * @since ??
	 *
	 * @param mixed $value The value to check and migrate.
	 *
	 * @return mixed The migrated value.
	 */
	private static function _migrate_attributes_recursive( $value ) {
		if ( is_string( $value ) ) {
			return self::_convert_global_color_css_variable( $value );
		}

		if ( is_array( $value ) ) {
			$migrated = [];
			foreach ( $value as $key => $item ) {
				$migrated[ $key ] = self::_migrate_attributes_recursive( $item );
			}
			return $migrated;
		}

		return $value;
	}

	/**
	 * Convert CSS variable global color to $variable() syntax.
	 *
	 * @since ??
	 *
	 * @param string $value The value to convert.
	 *
	 * @return string The converted value.
	 */
	private static function _convert_global_color_css_variable( string $value ): string {
		if ( ! preg_match( self::$_css_variable_pattern, $value, $matches ) ) {
			return $value;
		}

		$global_color_id = $matches[1]; // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.DeprecatedWhitelistCommentFound -- Now includes 'gcid-' prefix.

		// Convert to $variable() syntax.
		$variable_data = wp_json_encode(
			[
				'type'  => 'color',
				'value' => [
					'name'     => $global_color_id, // Keep the full 'gcid-' prefixed ID.
					'settings' => new \stdClass(), // Empty object for settings.
				],
			],
			JSON_UNESCAPED_SLASHES
		);

		return '$variable(' . $variable_data . ')$';
	}

	/**
	 * Convert flat module objects back to block array structure.
	 *
	 * @since ??
	 *
	 * @param array $flat_objects The flat module objects.
	 * @return array The block array structure.
	 */
	private static function _flat_objects_to_blocks( array $flat_objects ): array {
		// Find the root object.
		$root = null;
		foreach ( $flat_objects as $object ) {
			if ( isset( $object['parent'] ) && ( null === $object['parent'] || 'root' === $object['parent'] ) ) {
				$root = $object;
				break;
			}
		}
		if ( ! $root ) {
			return [];
		}
		return array_map(
			function( $child_id ) use ( $flat_objects ) {
				return self::_build_block_from_flat( $child_id, $flat_objects );
			},
			$root['children']
		);
	}

	/**
	 * Build a block from flat module object.
	 *
	 * @since ??
	 *
	 * @param string $id The module ID.
	 * @param array  $flat_objects All flat module objects.
	 * @return array The block array.
	 */
	private static function _build_block_from_flat( string $id, array $flat_objects ): array {
		$object = $flat_objects[ $id ];
		$block  = [
			'blockName'    => $object['name'],
			'attrs'        => $object['props']['attrs'] ?? [],
			'innerBlocks'  => [],
			'innerContent' => [],
		];
		if ( ! empty( $object['children'] ) ) {
			foreach ( $object['children'] as $child_id ) {
				$block['innerBlocks'][]  = self::_build_block_from_flat( $child_id, $flat_objects );
				$block['innerContent'][] = null; // Placeholder, will be filled by serializer.
			}
		}
		if ( isset( $object['props']['innerHTML'] ) ) {
			$block['innerContent'][] = $object['props']['innerHTML'];
		}
		return $block;
	}
}
